<?php
/* --------------------------------------------------------------
 PluginsLoader.php 2020-03-11
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Application\Plugins\Implementation;

use Gambio\Core\Application\ValueObjects\Path;
use Gambio\Core\Application\Plugins\AbstractPlugin;
use Gambio\Core\Application\Plugins\Plugin;
use PhpParser\Node;
use PhpParser\NodeFinder;
use PhpParser\Parser;
use PhpParser\ParserFactory;
use RecursiveIteratorIterator as Iterator;
use RecursiveDirectoryIterator as Directory;
use SplFileInfo;

/**
 * Class PluginsLoader
 * @package Gambio\Core\Application\Plugins
 */
class PluginsLoader
{
    private const VALID_IMPLEMENTATIONS = [
        Plugin::class,
        AbstractPlugin::class
    ];
    
    /**
     * @var Path
     */
    private $path;
    
    /**
     * @var Parser
     */
    private $parser;
    
    /**
     * @var NodeFinder
     */
    private $finder;
    
    
    /**
     * PluginsLoader constructor.
     *
     * @param ParserFactory $parserFactory
     * @param NodeFinder    $finder
     * @param Path          $path
     */
    public function __construct(Path $path, ParserFactory $parserFactory, NodeFinder $finder)
    {
        $this->path   = $path;
        $this->parser = $parserFactory->create(ParserFactory::ONLY_PHP7);
        $this->finder = $finder;
    }
    
    
    /**
     * Loads a list of plugins.
     * The result should be cached, because this is a very expensive operation (in terms of performance).
     *
     * @return array
     */
    public function load(): array
    {
        $gxModulesPath = "{$this->path->base()}/GXModules";
        $iterator      = new Iterator(new Directory($gxModulesPath, Directory::SKIP_DOTS), Iterator::LEAVES_ONLY);
        $plugins       = [];
        foreach ($iterator as $file) {
            /** @var SplFileInfo $file */
            
            if ($file->getExtension() === 'php') {
                try {
                    $content = file_get_contents($file->getPathname());
                    $ast     = $this->parser->parse($content);
                    
                    $class = $this->findClass($ast);
                    if (!$class || !$class->name) {
                        continue;
                    }
                    
                    $className = $class->name->toString();
                    $namespace = $this->findNamespace($ast);
                    $fqn       = $namespace ? "{$namespace}\\{$className}" : $className;
                    $useStmts  = $this->findUseStmts($ast);
                    
                    foreach (self::VALID_IMPLEMENTATIONS as $implementation) {
                        $implements = $class->implements;
                        
                        foreach ($implements as $implement) {
                            if ($this->isMatching($implement, $implementation, $useStmts)) {
                                $plugins[realpath($file->getPathname())] = $fqn;
                            }
                        }
                        
                        $extends = $class->extends;
                        if ($extends && $this->isMatching($extends, $implementation, $useStmts)) {
                            $plugins[realpath($file->getPathname())] = $fqn;
                        }
                    }
                } catch (\Exception $e) {
                    continue;
                } catch (\Throwable $e) {
                    continue;
                }
            }
        }
        
        return $plugins;
    }
    
    
    /**
     * Checks if node name matches implementation.
     *
     * Use statements and aliases are taken into account. The aliases will be determined in ::findUseStmts.
     *
     * @param Node\Name $name
     * @param string    $implementation
     * @param array     $useStatements
     *
     * @return bool
     * @see PluginsLoader::findUseStmts
     */
    private function isMatching(Node\Name $name, string $implementation, array $useStatements): bool
    {
        $nameString = $name->toString();
        if ($nameString === $implementation) {
            return true;
        }
        
        if (array_key_exists($nameString, $useStatements)
            && $useStatements[$nameString] === $implementation) {
            return true;
        }
        
        return false;
    }
    
    
    /**
     * Searches for the namespace name of the ast and return it as string, if found.
     *
     * @param array $ast
     *
     * @return string|null
     */
    private function findNamespace(array $ast): ?string
    {
        /** @var Node\Stmt\Namespace_ $namespace */
        $namespace = $this->finder->findFirstInstanceOf($ast, Node\Stmt\Namespace_::class);
        
        if ($namespace) {
            return $namespace->name->toString();
        }
        
        return null;
    }
    
    
    /**
     * Find use statements in ast.
     *
     * This method also takes aliases into account an provides them as key in the returned array.
     *
     * @param array $ast
     *
     * @return array
     */
    private function findUseStmts(array $ast): array
    {
        /** @var Node\Stmt\Use_[] $statements */
        $statements = $this->finder->findInstanceOf($ast, Node\Stmt\Use_::class);
        $uses       = [];
        
        foreach ($statements as $statement) {
            foreach ($statement->uses as $useUse) {
                $fqn = $useUse->name->toString();
                
                if ($useUse->alias) {
                    $uses[$useUse->alias->toString()] = $fqn;
                } else {
                    $namespaceSegments                   = explode('\\', $fqn);
                    $uses[array_pop($namespaceSegments)] = $fqn;
                }
            }
        }
        
        return $uses;
    }
    
    
    /**
     * Searches for the first class in the AST.
     *
     * @param array $ast
     *
     * @return Node\Stmt\Class_|null
     */
    private function findClass(array $ast): ?Node\Stmt\Class_
    {
        /** @var Node\Stmt\Class_ $result */
        $result = $this->finder->findFirstInstanceOf($ast, Node\Stmt\Class_::class);
        
        if (!$result) {
            return null;
        }
        if (!$result->extends && !$result->implements) {
            return null;
        }
        
        return $result;
    }
}